<?php
declare(strict_types=1);

function admin_list_users(PDO $db): array {
  return $db->query("SELECT id, username, email, role, can_read, can_write, created_at FROM users ORDER BY created_at DESC")->fetchAll();
}

function admin_create_user(PDO $db, string $username, string $email, string $password, string $role): void {
  $role = ($role === 'admin') ? 'admin' : 'user';
  $passHash = password_hash($password, PASSWORD_DEFAULT);
  $st = $db->prepare("INSERT INTO users(username,email,pass_hash,role,created_at) VALUES(?,?,?,?,?)");
  $st->execute([trim($username), trim($email), $passHash, $role, now_iso()]);
}

function admin_update_user_perms(PDO $db, int $userId, int $canRead, int $canWrite, string $role): void {
  $role = ($role === 'admin') ? 'admin' : 'user';
  $st = $db->prepare("UPDATE users SET can_read=?, can_write=?, role=? WHERE id=?");
  $st->execute([$canRead ? 1 : 0, $canWrite ? 1 : 0, $role, $userId]);
}

function admin_delete_user(PDO $db, array $config, int $userId): void {
  // Optionally delete files in storage
  if (!empty($config['delete_user_files'])) {
    $root = storage_user_root($config['storage_path'], $userId);
    if (is_dir($root)) {
      $it = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($root, FilesystemIterator::SKIP_DOTS),
        RecursiveIteratorIterator::CHILD_FIRST
      );
      foreach ($it as $f) {
        if ($f->isDir()) @rmdir($f->getPathname());
        else @unlink($f->getPathname());
      }
      @rmdir($root);
    }
  }
  $st = $db->prepare("DELETE FROM users WHERE id=?");
  $st->execute([$userId]);
}

function find_user_by_username(PDO $db, string $username): ?array {
  $st = $db->prepare("SELECT id, username, email FROM users WHERE username=? LIMIT 1");
  $st->execute([trim($username)]);
  $r = $st->fetch();
  return $r ?: null;
}

function search_users(PDO $db, string $q, int $limit=10): array {
  $q = trim($q);
  $st = $db->prepare("SELECT id, username, email FROM users WHERE username LIKE ? OR email LIKE ? ORDER BY username COLLATE NOCASE LIMIT ?");
  $like = '%' . $q . '%';
  $st->execute([$like,$like,$limit]);
  return $st->fetchAll();
}
