<?php
if (!ob_get_level()) { ob_start(); }
session_start();

function load_db_config(): array {
  $cfgFile = __DIR__ . '/config.php';

  if (is_file($cfgFile)) {
    $out = include $cfgFile;
    if (is_array($out)) {
      $dsn  = $out['dsn']  ?? $out['DB_DSN'] ?? $out['db_dsn'] ?? null;
      $user = $out['user'] ?? $out['DB_USER'] ?? $out['db_user'] ?? null;
      $pass = $out['pass'] ?? $out['DB_PASS'] ?? $out['db_pass'] ?? null;
      if ($dsn) return ['dsn'=>$dsn,'user'=>$user ?? '','pass'=>$pass ?? ''];
    }
  }

  if (defined('DB_HOST') && defined('DB_NAME')) {
    $dsn = "mysql:host=".(string)DB_HOST.";dbname=".(string)DB_NAME.";charset=utf8mb4";
    $user = defined('DB_USER') ? (string)DB_USER : '';
    $pass = defined('DB_PASS') ? (string)DB_PASS : '';
    return ['dsn'=>$dsn,'user'=>$user,'pass'=>$pass];
  }

  if (defined('DB_DSN')) {
    $dsn = (string)DB_DSN;
    $user = defined('DB_USER') ? (string)DB_USER : '';
    $pass = defined('DB_PASS') ? (string)DB_PASS : '';
    return ['dsn'=>$dsn,'user'=>$user,'pass'=>$pass];
  }

  $dsn = getenv('DB_DSN') ?: null;
  if ($dsn) return ['dsn'=>$dsn,'user'=>getenv('DB_USER')?:'', 'pass'=>getenv('DB_PASS')?:''];

  http_response_code(500);
  echo "<pre style='padding:14px;background:#fff3cd;border:1px solid #ffeeba;border-radius:10px'>".
       "Error de configuración de BD: DSN vacío o inválido.\n".
       "Revisa public_html/inc/config.php.\n".
       "</pre>";
  exit;
}

function db(): PDO {
  static $pdo = null;
  if ($pdo) return $pdo;
  $cfg = load_db_config();
  $pdo = new PDO($cfg['dsn'], $cfg['user'], $cfg['pass'], [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
  ]);
  return $pdo;
}

function csrf_token(): string {
  if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
  return $_SESSION['csrf'];
}
function csrf_check(): void {
  $t = $_POST['csrf'] ?? '';
  if (!$t || !hash_equals($_SESSION['csrf'] ?? '', $t)) {
    http_response_code(403);
    exit('CSRF inválido');
  }
}

function flash_set(string $type, string $msg): void { $_SESSION['flash'][] = ['type'=>$type,'msg'=>$msg]; }
function flash_get(): array { $f = $_SESSION['flash'] ?? []; $_SESSION['flash'] = []; return $f; }

function current_user(): ?array { return $_SESSION['user'] ?? null; }
function is_admin(): bool { return (bool)(current_user()['is_admin'] ?? false); }

function url(string $path): string {
  if (preg_match('#^https?://#i', $path)) return $path;
  return '/' . ltrim($path, '/');
}
function h($s): string { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

function redirect(string $path): void {
  $dest = url($path);
  if (!headers_sent()) {
    header('Location: ' . $dest);
    exit;
  }
  echo "<script>location.href=" . json_encode($dest) . ";</script>";
  echo "<noscript><meta http-equiv='refresh' content='0;url=" . h($dest) . "'></noscript>";
  exit;
}

function require_login(): void {
  if (!current_user()) { redirect('login.php'); }
  ensure_schema_safe();
}

function eur($n): string { return number_format((float)$n,2,',','.') . " €"; }

function table_exists(string $table): bool {
  try { $st = db()->prepare("SHOW TABLES LIKE ?"); $st->execute([$table]); return (bool)$st->fetchColumn(); }
  catch (Throwable $e) { return false; }
}
function table_has_column(string $table, string $col): bool {
  try { $pdo=db(); $st = $pdo->query("SHOW COLUMNS FROM `$table` LIKE " . $pdo->quote($col)); return (bool)$st->fetch(); }
  catch (Throwable $e) { return false; }
}

function generate_customer_no(): string {
  return (string)random_int(10000000, 99999999);
}

function ensure_schema_safe(): void {
  if (!empty($_SESSION['schema_ok_v4'])) return;
  try {
    $pdo = db();

    if (table_exists('users') && !table_has_column('users','avatar_url')) {
      $pdo->exec("ALTER TABLE users ADD COLUMN avatar_url VARCHAR(255) NULL");
    }

    if (table_exists('company_settings') && !table_has_column('company_settings','logo_path')) {
      $pdo->exec("ALTER TABLE company_settings ADD COLUMN logo_path VARCHAR(255) NULL");
    }

    if (table_exists('clients')) {
      if (!table_has_column('clients','customer_no')) {
        $pdo->exec("ALTER TABLE clients ADD COLUMN customer_no VARCHAR(30) NULL");
      }
      if (!table_has_column('clients','postal_code')) $pdo->exec("ALTER TABLE clients ADD COLUMN postal_code VARCHAR(10) NULL");
      if (!table_has_column('clients','localidad'))   $pdo->exec("ALTER TABLE clients ADD COLUMN localidad VARCHAR(120) NULL");
      if (!table_has_column('clients','provincia'))   $pdo->exec("ALTER TABLE clients ADD COLUMN provincia VARCHAR(120) NULL");

      if (table_has_column('clients','customer_no')) {
        $rows = $pdo->query("SELECT id FROM clients WHERE customer_no IS NULL OR customer_no='' LIMIT 200")->fetchAll();
        if ($rows) {
          $up = $pdo->prepare("UPDATE clients SET customer_no=? WHERE id=?");
          foreach($rows as $r){
            $up->execute([generate_customer_no(), (int)$r['id']]);
          }
        }
      }
    }

    if (table_exists('documents')) {
      if (!table_has_column('documents','rectifies_invoice_id')) $pdo->exec("ALTER TABLE documents ADD COLUMN rectifies_invoice_id INT NULL");
      if (!table_has_column('documents','is_rectificativa')) $pdo->exec("ALTER TABLE documents ADD COLUMN is_rectificativa TINYINT(1) NOT NULL DEFAULT 0");
    }

    $pdo->exec("
      CREATE TABLE IF NOT EXISTS client_files(
        id INT AUTO_INCREMENT PRIMARY KEY,
        client_id INT NOT NULL,
        original_name VARCHAR(255) NOT NULL,
        stored_path VARCHAR(255) NOT NULL,
        uploaded_by INT NULL,
        uploaded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_client (client_id)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ");

    $_SESSION['schema_ok_v4']=1;
  } catch (Throwable $e) {
    $_SESSION['schema_ok_v4']=1;
  }
}

function avatar_url(array $user): string {
  $u = trim((string)($user['avatar_url'] ?? ''));
  if ($u !== '') return $u;
  $seed = rawurlencode((string)($user['username'] ?? 'usuario'));
  return "https://api.dicebear.com/7.x/notionists/svg?seed={$seed}";
}
