<?php
if (!ob_get_level()) { ob_start(); }
session_start();

function load_db_config(): array {
  $cfgFile = __DIR__ . '/config.php';
  if (is_file($cfgFile)) {
    $out = include $cfgFile;
    if (is_array($out)) {
      $dsn  = $out['dsn']  ?? $out['DB_DSN'] ?? $out['db_dsn'] ?? null;
      $user = $out['user'] ?? $out['DB_USER'] ?? $out['db_user'] ?? null;
      $pass = $out['pass'] ?? $out['DB_PASS'] ?? $out['db_pass'] ?? null;
      if ($dsn) return ['dsn'=>$dsn,'user'=>$user ?? '','pass'=>$pass ?? ''];
    }
  }
  if (defined('DB_HOST') && defined('DB_NAME')) {
    $dsn = "mysql:host=".(string)DB_HOST.";dbname=".(string)DB_NAME.";charset=utf8mb4";
    $user = defined('DB_USER') ? (string)DB_USER : '';
    $pass = defined('DB_PASS') ? (string)DB_PASS : '';
    return ['dsn'=>$dsn,'user'=>$user,'pass'=>$pass];
  }
  $dsn = getenv('DB_DSN') ?: null;
  if ($dsn) return ['dsn'=>$dsn,'user'=>getenv('DB_USER')?:'', 'pass'=>getenv('DB_PASS')?:''];
  http_response_code(500);
  echo "<pre style='padding:14px;background:#fff3cd;border:1px solid #ffeeba;border-radius:10px'>".
       "Error de configuración de BD: DSN vacío o inválido.\nRevisa public_html/inc/config.php.\n</pre>";
  exit;
}

function db(): PDO {
  static $pdo = null;
  if ($pdo) return $pdo;
  $cfg = load_db_config();
  $pdo = new PDO($cfg['dsn'], $cfg['user'], $cfg['pass'], [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
  ]);
  return $pdo;
}

function csrf_token(): string { if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16)); return $_SESSION['csrf']; }
function csrf_check(): void {
  $t = $_POST['csrf'] ?? '';
  if (!$t || !hash_equals($_SESSION['csrf'] ?? '', $t)) { http_response_code(403); exit('CSRF inválido'); }
}

function flash_set(string $type, string $msg): void { $_SESSION['flash'][] = ['type'=>$type,'msg'=>$msg]; }
function flash_get(): array { $f = $_SESSION['flash'] ?? []; $_SESSION['flash'] = []; return $f; }

function current_user(): ?array { return $_SESSION['user'] ?? null; }
function current_role(): string { return (string)(current_user()['role'] ?? 'staff'); }
function is_admin(): bool { return current_role() === 'admin' || (bool)(current_user()['is_admin'] ?? false); }
function is_client_user(): bool { return current_role() === 'client'; }

function url(string $path): string {
  if (preg_match('#^https?://#i', $path)) return $path;
  return '/' . ltrim($path, '/');
}
function h($s): string { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

function redirect(string $path): void {
  $dest = url($path);
  if (!headers_sent()) { header('Location: ' . $dest); exit; }
  echo "<script>location.href=" . json_encode($dest) . ";</script>";
  echo "<noscript><meta http-equiv='refresh' content='0;url=" . h($dest) . "'></noscript>";
  exit;
}

function require_login(): void {
  if (!current_user()) { redirect('login.php'); }
  ensure_schema_safe();
}

function eur($n): string { return number_format((float)$n,2,',','.') . " €"; }

function table_exists(string $table): bool {
  try { $st = db()->prepare("SHOW TABLES LIKE ?"); $st->execute([$table]); return (bool)$st->fetchColumn(); }
  catch (Throwable $e) { return false; }
}
function table_has_column(string $table, string $col): bool {
  try { $pdo=db(); $st = $pdo->query("SHOW COLUMNS FROM `$table` LIKE " . $pdo->quote($col)); return (bool)$st->fetch(); }
  catch (Throwable $e) { return false; }
}

function generate_customer_no(): string { return (string)random_int(10000000, 99999999); }

function client_customer_col(): ?string {
  if (!table_exists('clients')) return null;
  if (table_has_column('clients','customer_number')) return 'customer_number';
  if (table_has_column('clients','customer_no')) return 'customer_no';
  return null;
}

function ensure_schema_safe(): void {
  if (!empty($_SESSION['schema_ok_v6'])) return;
  try {
    $pdo = db();

    if (table_exists('users')) {
      if (!table_has_column('users','avatar_url')) $pdo->exec("ALTER TABLE users ADD COLUMN avatar_url VARCHAR(255) NULL");
      if (!table_has_column('users','role')) $pdo->exec("ALTER TABLE users ADD COLUMN role VARCHAR(20) NOT NULL DEFAULT 'staff'");
      if (!table_has_column('users','client_id')) $pdo->exec("ALTER TABLE users ADD COLUMN client_id INT NULL");
    }

    if (table_exists('company_settings') && !table_has_column('company_settings','logo_path')) {
      $pdo->exec("ALTER TABLE company_settings ADD COLUMN logo_path VARCHAR(255) NULL");
    }

    if (table_exists('clients')) {
      if (!table_has_column('clients','email')) $pdo->exec("ALTER TABLE clients ADD COLUMN email VARCHAR(190) NULL");
      if (!table_has_column('clients','postal_code')) $pdo->exec("ALTER TABLE clients ADD COLUMN postal_code VARCHAR(10) NULL");
      if (!table_has_column('clients','localidad'))   $pdo->exec("ALTER TABLE clients ADD COLUMN localidad VARCHAR(120) NULL");
      if (!table_has_column('clients','provincia'))   $pdo->exec("ALTER TABLE clients ADD COLUMN provincia VARCHAR(120) NULL");

      if (!table_has_column('clients','customer_number') && !table_has_column('clients','customer_no')) {
        $pdo->exec("ALTER TABLE clients ADD COLUMN customer_no VARCHAR(30) NULL");
      }
      $cc = client_customer_col();
      if ($cc) {
        $rows = $pdo->query("SELECT id FROM clients WHERE `$cc` IS NULL OR `$cc`='' LIMIT 200")->fetchAll();
        if ($rows) {
          $up = $pdo->prepare("UPDATE clients SET `$cc`=? WHERE id=?");
          foreach($rows as $r){ $up->execute([generate_customer_no(), (int)$r['id']]); }
        }
      }
    }

    $pdo->exec("
      CREATE TABLE IF NOT EXISTS messages(
        id INT AUTO_INCREMENT PRIMARY KEY,
        client_id INT NOT NULL,
        from_user_id INT NULL,
        subject VARCHAR(190) NOT NULL,
        body TEXT NOT NULL,
        is_read TINYINT(1) NOT NULL DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_client (client_id),
        INDEX idx_read (client_id,is_read)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ");

    $pdo->exec("
      CREATE TABLE IF NOT EXISTS client_files(
        id INT AUTO_INCREMENT PRIMARY KEY,
        client_id INT NOT NULL,
        original_name VARCHAR(255) NOT NULL,
        stored_path VARCHAR(255) NOT NULL,
        uploaded_by INT NULL,
        uploaded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_client (client_id)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ");

    $_SESSION['schema_ok_v6']=1;
  } catch (Throwable $e) {
    $_SESSION['schema_ok_v6']=1;
  }
}

function avatar_url(array $user): string {
  $u = trim((string)($user['avatar_url'] ?? ''));
  if ($u !== '') return $u;
  $seed = rawurlencode((string)($user['username'] ?? 'usuario'));
  return "https://api.dicebear.com/7.x/notionists/svg?seed={$seed}";
}

function normalize_phone_digits(string $p): string { return preg_replace('/[^0-9+]/','', $p) ?: $p; }
function whatsapp_link(string $p): string {
  $d = preg_replace('/[^0-9]/','', $p);
  return $d ? "https://wa.me/".$d : '#';
}

function send_mail_with_pdf(string $to, string $subject, string $bodyText, string $pdfBytes, string $filename, string $from=null): bool {
  $boundary = "crm_" . bin2hex(random_bytes(8));
  $from = $from ?: ("CRM <no-reply@" . ($_SERVER['SERVER_NAME'] ?? 'localhost') . ">");
  $headers = [];
  $headers[] = "From: " . $from;
  $headers[] = "MIME-Version: 1.0";
  $headers[] = "Content-Type: multipart/mixed; boundary=\"{$boundary}\"";

  $msg = "--{$boundary}\r\n";
  $msg .= "Content-Type: text/plain; charset=UTF-8\r\n";
  $msg .= "Content-Transfer-Encoding: 8bit\r\n\r\n";
  $msg .= $bodyText . "\r\n\r\n";

  $b64 = chunk_split(base64_encode($pdfBytes));
  $msg .= "--{$boundary}\r\n";
  $msg .= "Content-Type: application/pdf; name=\"{$filename}\"\r\n";
  $msg .= "Content-Transfer-Encoding: base64\r\n";
  $msg .= "Content-Disposition: attachment; filename=\"{$filename}\"\r\n\r\n";
  $msg .= $b64 . "\r\n";
  $msg .= "--{$boundary}--";

  return @mail($to, "=?UTF-8?B?".base64_encode($subject)."?=", $msg, implode("\r\n",$headers));
}
