<?php
// Evita "headers already sent" si algún fichero imprime espacios/BOM accidentalmente
if (!ob_get_level()) { ob_start(); }

session_start();

/**
 * Carga config de BD compatible con varios formatos:
 * - config.php define constantes DB_HOST/DB_NAME/DB_USER/DB_PASS
 * - o define DB_DSN/DB_USER/DB_PASS
 * - o devuelve un array con ['dsn','user','pass']
 */
function load_db_config(): array {
  $cfgFile = __DIR__ . '/config.php';

  if (is_file($cfgFile)) {
    $out = include $cfgFile;
    if (is_array($out)) {
      $dsn  = $out['dsn']  ?? $out['DB_DSN'] ?? $out['db_dsn'] ?? null;
      $user = $out['user'] ?? $out['DB_USER'] ?? $out['db_user'] ?? null;
      $pass = $out['pass'] ?? $out['DB_PASS'] ?? $out['db_pass'] ?? null;
      if ($dsn) return ['dsn'=>$dsn,'user'=>$user ?? '','pass'=>$pass ?? ''];
    }
  }

  if (defined('DB_HOST') && defined('DB_NAME')) {
    $host = (string)DB_HOST;
    $name = (string)DB_NAME;
    $dsn = "mysql:host={$host};dbname={$name};charset=utf8mb4";
    $user = defined('DB_USER') ? (string)DB_USER : '';
    $pass = defined('DB_PASS') ? (string)DB_PASS : '';
    return ['dsn'=>$dsn,'user'=>$user,'pass'=>$pass];
  }

  if (defined('DB_DSN')) {
    $dsn = (string)DB_DSN;
    $user = defined('DB_USER') ? (string)DB_USER : '';
    $pass = defined('DB_PASS') ? (string)DB_PASS : '';
    return ['dsn'=>$dsn,'user'=>$user,'pass'=>$pass];
  }

  $dsn = getenv('DB_DSN') ?: null;
  if ($dsn) return ['dsn'=>$dsn,'user'=>getenv('DB_USER')?:'', 'pass'=>getenv('DB_PASS')?:''];

  http_response_code(500);
  echo "<pre style='padding:14px;background:#fff3cd;border:1px solid #ffeeba;border-radius:10px'>".
       "Error de configuración de BD: DSN vacío o inválido.\n".
       "Revisa public_html/inc/config.php.\n".
       "</pre>";
  exit;
}

function db(): PDO {
  static $pdo = null;
  if ($pdo) return $pdo;

  $cfg = load_db_config();
  $pdo = new PDO($cfg['dsn'], $cfg['user'], $cfg['pass'], [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
  ]);
  return $pdo;
}

function csrf_token(): string {
  if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
  return $_SESSION['csrf'];
}
function csrf_check(): void {
  $t = $_POST['csrf'] ?? '';
  if (!$t || !hash_equals($_SESSION['csrf'] ?? '', $t)) {
    http_response_code(403);
    exit('CSRF inválido');
  }
}

function flash_set(string $type, string $msg): void { $_SESSION['flash'][] = ['type'=>$type,'msg'=>$msg]; }
function flash_get(): array { $f = $_SESSION['flash'] ?? []; $_SESSION['flash'] = []; return $f; }

function current_user(): ?array { return $_SESSION['user'] ?? null; }
function is_admin(): bool { return (bool)(current_user()['is_admin'] ?? false); }

function url(string $path): string {
  if (preg_match('#^https?://#i', $path)) return $path;
  return '/' . ltrim($path, '/');
}

function h($s): string { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

/**
 * Redirección robusta: si ya se enviaron headers, usa JS/meta.
 */
function redirect(string $path): void {
  $dest = url($path);
  if (!headers_sent()) {
    header('Location: ' . $dest);
    exit;
  }
  echo "<script>location.href=" . json_encode($dest) . ";</script>";
  echo "<noscript><meta http-equiv='refresh' content='0;url=" . h($dest) . "'></noscript>";
  exit;
}

function require_login(): void {
  if (!current_user()) { redirect('login.php'); }
  ensure_schema_safe();
}

// --- Schema helpers ---
function table_exists(string $table): bool {
  try { $st = db()->prepare("SHOW TABLES LIKE ?"); $st->execute([$table]); return (bool)$st->fetchColumn(); }
  catch (Throwable $e) { return false; }
}
function table_has_column(string $table, string $col): bool {
  try { $pdo=db(); $st = $pdo->query("SHOW COLUMNS FROM `$table` LIKE " . $pdo->quote($col)); return (bool)$st->fetch(); }
  catch (Throwable $e) { return false; }
}
function ensure_schema_safe(): void {
  if (!empty($_SESSION['schema_ok_v2'])) return;
  try {
    $pdo = db();
    if (table_exists('users') && !table_has_column('users','avatar_url')) {
      $pdo->exec("ALTER TABLE users ADD COLUMN avatar_url VARCHAR(255) NULL");
    }
    if (table_exists('company_settings') && !table_has_column('company_settings','logo_path')) {
      $pdo->exec("ALTER TABLE company_settings ADD COLUMN logo_path VARCHAR(255) NULL");
    }
    if (table_exists('clients')) {
      if (!table_has_column('clients','postal_code')) $pdo->exec("ALTER TABLE clients ADD COLUMN postal_code VARCHAR(10) NULL");
      if (!table_has_column('clients','localidad'))   $pdo->exec("ALTER TABLE clients ADD COLUMN localidad VARCHAR(120) NULL");
      if (!table_has_column('clients','provincia'))   $pdo->exec("ALTER TABLE clients ADD COLUMN provincia VARCHAR(120) NULL");
    }
    $_SESSION['schema_ok_v2']=1;
  } catch (Throwable $e) {
    $_SESSION['schema_ok_v2']=1;
  }
}

function avatar_url(array $user): string {
  $u = trim((string)($user['avatar_url'] ?? ''));
  if ($u !== '') return $u;
  $seed = rawurlencode((string)($user['username'] ?? 'usuario'));
  return "https://api.dicebear.com/7.x/notionists/svg?seed={$seed}";
}
